//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft shared
// source or premium shared source license agreement under which you licensed
// this source code. If you did not accept the terms of the license agreement,
// you are not authorized to use this source code. For the terms of the license,
// please see the license agreement between you and Microsoft or, if applicable,
// see the SOURCE.RTF on your install media or the root of your tools installation.
// THE SOURCE CODE IS PROVIDED "AS IS", WITH NO WARRANTIES.
//
#include "Geometry.h"
#include "Permutations.h"
#include "ImageManagement.h"
#include "qamath.h"
#include "Transforms.h"
#include "TransformWrapper.h"
#include "ShearTest.h"
#include <d3dm.h>
#include <tux.h>
#include "DebugOutput.h"

TESTPROCAPI ShearTest(LPDIRECT3DMOBILEDEVICE pDevice,
                      D3DMTRANSFORMSTATETYPE TransMatrix,
                      D3DQA_SHEARTYPE ShearType,
                      UINT uiFrameNumber,
                      D3DMFORMAT Format,
                      UINT uiTestID, 
                      WCHAR *wszImageDir,
                      UINT uiInstanceHandle,
                      HWND hWnd,
                      LPTESTCASEARGS pTestCaseArgs)
{
    HRESULT hr;
    
	//
	// Level of detail in sphere model
	//
	CONST UINT uiNumDivisions = 12;

	//
	// Shear factor
	//
	float fShear;

	//
	// Each test case consists of multiple rendered frames
	//
	UINT uiFrame;

	//
	// Shear matrix to be applied to Direct3D TnL
	//
	D3DMMATRIX ShearMat;

	//
	// Interfaces for scene data
	//
	LPDIRECT3DMOBILEVERTEXBUFFER pVBSphere = NULL;
	LPDIRECT3DMOBILEINDEXBUFFER pIBSphere = NULL;

	//
	// All failure conditions set this to an error
	//
	INT iTestResult = TPR_PASS;

	//
	// Number of frames expected to be generated by this test case
	//
	UINT uiNumDesiredFrames;


	//
	// The geometry-generating algorithm reports it's "specs"; useful at
	// DrawPrimitive-time
	//
	D3DMPRIMITIVETYPE PrimType;
	UINT uiPrimCount;
	UINT uiNumVertices;

	UINT uiTotalFrames;

	//
	// Shear loop does not have nested iterators
	//
	CONST UINT uiNumIters = 1; 

	//
	// Flat shading is in base profile
	//
	pDevice->SetRenderState( D3DMRS_SHADEMODE, D3DMSHADE_FLAT );

	//
	// Lighting is "baked" in verts
	//
	pDevice->SetRenderState( D3DMRS_LIGHTING, 0 );

	pDevice->SetRenderState(D3DMRS_CULLMODE, D3DMCULL_CW);

	//
	// Enable clipping
	//
	if( FAILED( hr = pDevice->SetRenderState(D3DMRS_CLIPPING, TRUE)))
	{
		DebugOut(DO_ERROR, L"SetRenderState(D3DMRS_CLIPPING,TRUE) failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}


	//
	// Retrieve expected number of frames for this kind of test case
	//
	uiNumDesiredFrames = NumTransformFramesByIters[uiNumIters];

	if (FAILED(hr = GetPermutationSpecs(1,                        // UINT uiNumIters,
	                                    uiNumDesiredFrames,       // UINT uiMaxIter,
	                                    &uiTotalFrames,           // UINT *puiNumStepsTotal,
	                                    NULL)))
	{
		DebugOut(DO_ERROR, L"GetPermutationSpecs failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	//
	// Confirm that test case can generate exactly the desired number of frames
	//
	if (uiNumDesiredFrames != uiTotalFrames)
	{
		DebugOut(DO_ERROR, 
		    L"Unexpected number of frames. (Expected %d, Got %d) Failing.",
		    uiNumDesiredFrames,
		    uiTotalFrames);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	if (FAILED(hr = GenerateSphere( pDevice,           // LPDIRECT3DMOBILEDEVICE pDevice,
	                                &pVBSphere,        // LPDIRECT3DMOBILEVERTEXBUFFER *ppVB,
	                                Format,            // D3DMFORMAT Format,
	                                &pIBSphere,        // LPDIRECT3DMOBILEINDEXBUFFER *ppIB, 
	                                uiNumDivisions,    // const int nDivisions
	                                &PrimType,         // D3DMPRIMITIVETYPE *pPrimType
	                                &uiPrimCount,      // PUINT puiPrimCount
	                                &uiNumVertices)))  // PUINT puiNumVertices
	{
		DebugOut(DO_ERROR, L"GenerateSphere failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	uiFrame = uiFrameNumber;

	if (FAILED(hr = SetTransformDefaults(pDevice))) // LPDIRECT3DMOBILEDEVICE pDevice
	{
		DebugOut(DO_ERROR, L"SetTransformDefaults failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	if (FAILED(hr = OneAxisPermutations(uiFrame,                 // UINT uiFrame,
		                                D3DQA_SHEARFACTOR_MIN,   // float fMin,
		                                D3DQA_SHEARFACTOR_MAX,   // float fMax,
		                                uiNumDesiredFrames,      // UINT uiMaxIters,
		                                &fShear)))               // float *pfAxis,
	{
		DebugOut(DO_ERROR, L"OneAxisPermutations failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	ShearMat = ShearMatrix( ShearType, // const D3DQA_SHEARTYPE ShearType,
			                fShear);   // float ShearFactor

	if (FAILED(hr = pDevice->SetTransform(TransMatrix,  // D3DMTRANSFORMSTATETYPE State,
								          &ShearMat, D3DMFMT_D3DMVALUE_FLOAT)))  // CONST D3DMMATRIX *pMatrix
	{
		DebugOut(DO_ERROR, L"SetTransform failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	//
	// Clear the backbuffer and the zbuffer
	//
	pDevice->Clear( 0, NULL, D3DMCLEAR_TARGET|D3DMCLEAR_ZBUFFER,
						D3DMCOLOR_XRGB(0,0,0), 1.0f, 0 );

	if (FAILED(hr = pDevice->BeginScene()))
	{
		DebugOut(DO_ERROR, L"BeginScene failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	if (FAILED(hr = pDevice->DrawIndexedPrimitive(
						 PrimType,     // D3DMPRIMITIVETYPE Type,
						 0,            // INT BaseVertexIndex,
						 0,            // UINT MinIndex,
						 uiNumVertices,// UINT NumVertices,
						 0,            // UINT StartIndex,
						 uiPrimCount)))// UINT PrimitiveCount
	{
		DebugOut(DO_ERROR, L"DrawIndexedPrimitive failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	if (FAILED(hr = pDevice->EndScene()))
	{
		DebugOut(DO_ERROR, L"EndScene failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	if (FAILED(hr = pDevice->Present(NULL, NULL, NULL, NULL)))
	{
		DebugOut(DO_ERROR, L"Present failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

	//
	// Flush the swap chain and capture a frame
	//
	if (FAILED(hr = DumpFlushedFrame(pTestCaseArgs, // LPTESTCASEARGS pTestCaseArgs
	                                 0,             // UINT uiFrameNumber,
	                                 NULL)))        // RECT *pRect = NULL
	{
		DebugOut(DO_ERROR, L"DumpFlushedFrame failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
		goto cleanup;
	}

cleanup:

	if (FAILED(hr = pDevice->SetIndices(NULL)))
	{
		DebugOut(DO_ERROR, L"SetIndices failed. (hr = 0x%08x) Failing.", hr);
		iTestResult = TPR_FAIL;
	}

	if (pVBSphere)
		pVBSphere->Release();

	if (pIBSphere)
		pIBSphere->Release();

	return iTestResult;
}


