//==============================================================================
// SimOrder.h
// Copyright (c) 2007 Ensemble Studios
//==============================================================================

#pragma once
#include "EntityID.h"
#include "Poolable.h"
#include "SimTarget.h"
#include "SimTypes.h"

//==============================================================================
//==============================================================================
class BSimOrder : public IPoolable
{
   public:
      enum
      {
         cTypeNone=0,
         cTypeMove,
         cTypeAttack,
         cTypeGather,
         cTypeRepair,
         cTypeRepairOther,
         cTypeRequestRepair,
         cTypeBuild,
         cTypeCapture,
         cTypeJoin,
         cTypeGarrison,
         cTypeUngarrison,
         cTypeDetonate,
         cTypeHonk,
         cTypeGuard,
         cTypeChangeMode,
         cTypeMines,
         cTypePlayBlockingAnimation,
         cTypeRallyPoint,
         cTypeIdle,
         cTypeUnpack,
         cTypeHitch,
         cTypeUnhitch,
         cTypeTransport,
         cTypeWander,
		   cTypeCloak,
         cTypeJump,
         cTypeJumpGather,
         cTypeJumpGarrison,
         cTypeJumpAttack,
         cTypePointBlankAttack,
         cTypeEnergyShield,
         cTypeJumpPull,
         cTypeInfantryEnergyShield,
         cNumberSimOrderTypes,
      };
      enum
      {
         cPrioritySim=0,
         cPriorityTrigger,
         cPriorityUser
      };
      enum
      {
         cJumpOrderJump,
         cJumpOrderJumpGather,
         cJumpOrderJumpGarrison,
         cJumpOrderJumpAttack,
         cJumpOrderJumpPull,
         cNumJumpOrders
      };

      BSimOrder() { }
      BSimOrder(const BSimOrder& ref)
      {
         mOwnerID=ref.mOwnerID;
         mID=ref.mID;
         mTarget=ref.mTarget;
         uint count=ref.mWaypoints.getSize();
         if (count > 0)
         {
            mWaypoints.setNumber(count);
            for (uint i=0; i < count; i++)
               mWaypoints[i]=ref.mWaypoints[i];
         }
         mRefCount=ref.mRefCount;
         mMode=ref.mMode;
         mAngle=ref.mAngle;
         mPriority=ref.mPriority;
         mAttackMove=ref.mAttackMove;
         mFlagOverridePosition = ref.mFlagOverridePosition;
         mFlagOverrideRange = ref.mFlagOverrideRange;
      };
      ~BSimOrder() { }

      //Init.
      virtual bool               init();
      virtual void               onAcquire() { }
      virtual void               onRelease() { }

      //Owner.
      BEntityID                  getOwnerID() const { return (mOwnerID); }
      void                       setOwnerID(BEntityID v) { mOwnerID=v; }
      //ID.
      uint                       getID() const { return (mID); }
      void                       setID(uint v) { mID=v; }
      //Type.
      const char*                getTypeName(BSimOrderType type) const;
      //Target.
      BSimTarget                 getTarget() const { return (mTarget); }
      void                       setTarget(BSimTarget v) { mTarget=v; }
      //Waypoints.
      uint                       getWaypointNumber() const { return (mWaypoints.getSize()); }
      BVector                    getWaypoint(uint index) const { if (index >= mWaypoints.getSize()) return (cInvalidVector); return (mWaypoints[index]); }
      const BDynamicSimVectorArray& getWaypoints() const { return (mWaypoints); }
      bool                       setWaypoints(const BDynamicVectorArray& waypoints);
      bool                       setWaypoints(const BDynamicSimVectorArray& waypoints);
      //Priority.
      uint                       getPriority() const { return (mPriority); }
      void                       setPriority(uint v) { mPriority=v; }
      //Mode.
      int8                       getMode() const  { return mMode; }
      void                       setMode(int8 v)  { mMode=v; }
      //Angle.
      float                      getAngle() const { return mAngle; }
      void                       setAngle(float v) { mAngle=v; }
      
      //RefCount.
      uint                       getRefCount() const { return (mRefCount); }      
      void                       incrementRefCount() { mRefCount++; }
      void                       decrementRefCount() { mRefCount--; }
      void                       zeroRefCount() { mRefCount=0; }

      //Attack Move.
      bool                       getAttackMove() const { return(mAttackMove); }
      void                       setAttackMove(bool v) { mAttackMove=v; }
      bool                       getAutoGeneratedAttackMove() const { return mAutoGeneratedAttackMove; }
      void                       setAutoGeneratedAttackMove(bool v) { mAutoGeneratedAttackMove = v; }

      // Override squad plotter
      bool                       getFlagOverridePosition() const { return (mFlagOverridePosition); }
      void                       setFlagOverridePosition(bool v) { mFlagOverridePosition = v; }
      bool                       getFlagOverrideRange() const { return (mFlagOverrideRange); }
      void                       setFlagOverrideRange(bool v) { mFlagOverrideRange = v; }

      //Operators.
      bool                       operator==(const BSimOrder& order) const;
      bool                       operator!=(const BSimOrder& order) const;
      
      bool                       isSimilarOrder(BSimOrder* pOrder);
      void                       copyFromTemplate(const BSimOrder& ref);
      

      DECLARE_FREELIST(BSimOrder, 5);

      GFDECLAREVERSION();
      bool save(BStream* pStream, int saveType) const;
      bool load(BStream* pStream, int saveType);

   protected:
      BSimTarget                 mTarget;
      BDynamicSimVectorArray     mWaypoints;
      BEntityID                  mOwnerID;

      uint                       mID;
      uint                       mRefCount;

      float                      mAngle;

      int8                       mMode;

      uint8                      mPriority:2;
      bool                       mAttackMove:1;
      bool                       mAutoGeneratedAttackMove:1;

      // Override squad plotter
      bool                       mFlagOverridePosition:1;
      bool                       mFlagOverrideRange:1;
};

typedef BSmallDynamicSimArray<BSimOrder*> BSimOrderArray;




//==============================================================================
//==============================================================================
class BSimOrderEntry
{
   public:
      enum
      {
         cStateNone=0,
         cStateQueued,
         cStatePaused,
         cStateExecute,
      };
   
      BSimOrderEntry() { init(); }
      BSimOrderEntry(const BSimOrderEntry& ref)
      {
         mOrder=ref.mOrder;
         mType=ref.mType;
         mState=ref.mState;
         mFlagRemoveMe=ref.mFlagRemoveMe;
         mFlagRemoveMeWhenInterruptible=ref.mFlagRemoveMeWhenInterruptible;
      };
      ~BSimOrderEntry() { }

      //Init.
      void                       init()
                                 {
                                    mOrder=NULL;
                                    mType=BSimOrder::cTypeNone;
                                    mState=cStateNone;
                                    mFlagRemoveMe=false;
                                    mFlagRemoveMeWhenInterruptible=false;
                                 }

      //Order.
      BSimOrder*                 getOrder() const { return (mOrder); }
      void                       setOrder(BSimOrder* v) { mOrder=v; }
      //Type.
      BSimOrderType              getType() const { return (mType); }
      bool                       isType(BSimOrderType v) const { return (mType == v); }
      void                       setType(BSimOrderType v) { mType=v; }
      const char*                getTypeName() const { return (mOrder->getTypeName(mType)); }
      //State.
      uint                       getState() const { return (mState); }
      void                       setState(uint v) { mState=v; }
      const char*                getStateName() const;

      //Delayed Removal
      bool                       getFlagRemoveMe() const { return(mFlagRemoveMe); }
      void                       setFlagRemoveMe(bool v) { mFlagRemoveMe=v; }

      //Delayed Removal for unInterruptible
      bool                       getFlagRemoveMeWhenInterruptible() const { return(mFlagRemoveMeWhenInterruptible); }
      void                       setFlagRemoveMeWhenInterruptible(bool v) { mFlagRemoveMeWhenInterruptible=v; }

      bool                       hasRemovalPending() const { return mFlagRemoveMe || mFlagRemoveMeWhenInterruptible; }

      //Operators.
      bool                       operator==(const BSimOrderEntry& v) const;
      bool                       operator!=(const BSimOrderEntry& v) const;

      GFDECLAREVERSION();
      bool save(BStream* pStream, int saveType) const;
      bool load(BStream* pStream, int saveType);

   protected:
      BSimOrder*                 mOrder;
      BSimOrderType              mType;
      uint                       mState:3;
      bool                       mFlagRemoveMe:1;
      bool                       mFlagRemoveMeWhenInterruptible:1;
};

typedef BSmallDynamicSimArray<BSimOrderEntry> BSimOrderEntryArray;
